/**
 * Contract Editor - Main JavaScript File
 * Professional Contract Editor with Drag & Drop functionality
 */

// Global Variables
let contractData = {
    content: '',
    variables: {},
    history: [],
    historyIndex: -1
};

let isDragging = false;
let draggedElement = null;

// DOM Elements
const elements = {
    // Toolbar buttons
    newContractBtn: document.getElementById('newContractBtn'),
    openBtn: document.getElementById('openBtn'),
    saveBtn: document.getElementById('saveBtn'),
    undoBtn: document.getElementById('undoBtn'),
    redoBtn: document.getElementById('redoBtn'),
    exportBtn: document.getElementById('exportBtn'),
    previewBtn: document.getElementById('previewBtn'),
    
    // Tabs
    tabBtns: document.querySelectorAll('.tab-btn'),
    tabContents: document.querySelectorAll('.tab-content'),
    
    // Search
    variableSearch: document.getElementById('variableSearch'),
    
    // Editor
    editorCanvas: document.getElementById('editorCanvas'),
    contractContent: document.querySelector('.contract-content'),
    
    // Variables and Elements
    variableItems: document.querySelectorAll('.variable-item'),
    elementItems: document.querySelectorAll('.element-item'),
    
    // Modals
    exportModal: document.getElementById('exportModal'),
    previewModal: document.getElementById('previewModal'),
    closeExportModal: document.getElementById('closeExportModal'),
    closePreviewModal: document.getElementById('closePreviewModal'),
    exportBtns: document.querySelectorAll('.export-btn'),
    previewContent: document.getElementById('previewContent')
};

// Initialize Application
document.addEventListener('DOMContentLoaded', function() {
    console.log('Contract Editor initialized');
    initializeEventListeners();
    initializeDragAndDrop();
    saveToHistory();
    updateToolbarState();
});

// Event Listeners
function initializeEventListeners() {
    // Toolbar Events
    elements.newContractBtn?.addEventListener('click', newContract);
    elements.openBtn?.addEventListener('click', openContract);
    elements.saveBtn?.addEventListener('click', saveContract);
    elements.undoBtn?.addEventListener('click', undo);
    elements.redoBtn?.addEventListener('click', redo);
    elements.exportBtn?.addEventListener('click', showExportModal);
    elements.previewBtn?.addEventListener('click', showPreview);
    
    // Tab Events
    elements.tabBtns.forEach(btn => {
        btn.addEventListener('click', () => switchTab(btn.dataset.tab));
    });
    
    // Search Events
    elements.variableSearch?.addEventListener('input', searchVariables);
    
    // Editor Events
    elements.contractContent?.addEventListener('input', handleContentChange);
    elements.contractContent?.addEventListener('paste', handlePaste);
    
    // Modal Events
    elements.closeExportModal?.addEventListener('click', hideExportModal);
    elements.closePreviewModal?.addEventListener('click', hidePreviewModal);
    elements.exportModal?.addEventListener('click', (e) => {
        if (e.target === elements.exportModal) hideExportModal();
    });
    elements.previewModal?.addEventListener('click', (e) => {
        if (e.target === elements.previewModal) hidePreviewModal();
    });
    
    // Export Events
    elements.exportBtns.forEach(btn => {
        btn.addEventListener('click', () => exportContract(btn.dataset.format));
    });
    
    // Keyboard Shortcuts
    document.addEventListener('keydown', handleKeyboardShortcuts);
}

// Drag and Drop Functionality
function initializeDragAndDrop() {
    // Make variable items draggable
    elements.variableItems.forEach(item => {
        item.draggable = true;
        item.addEventListener('dragstart', handleDragStart);
        item.addEventListener('dragend', handleDragEnd);
    });
    
    // Make element items draggable
    elements.elementItems.forEach(item => {
        item.draggable = true;
        item.addEventListener('dragstart', handleDragStart);
        item.addEventListener('dragend', handleDragEnd);
    });
    
    // Make editor canvas droppable
    if (elements.contractContent) {
        elements.contractContent.addEventListener('dragover', handleDragOver);
        elements.contractContent.addEventListener('drop', handleDrop);
        elements.contractContent.addEventListener('dragenter', handleDragEnter);
        elements.contractContent.addEventListener('dragleave', handleDragLeave);
    }
}

function handleDragStart(e) {
    isDragging = true;
    draggedElement = e.target;
    e.target.classList.add('dragging');
    
    // Set drag data
    if (e.target.classList.contains('variable-item')) {
        const variable = e.target.dataset.variable;
        const variableName = e.target.querySelector('.variable-name').textContent;
        e.dataTransfer.setData('text/plain', variable);
        e.dataTransfer.setData('application/json', JSON.stringify({
            type: 'variable',
            variable: variable,
            name: variableName
        }));
    } else if (e.target.classList.contains('element-item')) {
        const elementType = e.target.dataset.element;
        e.dataTransfer.setData('application/json', JSON.stringify({
            type: 'element',
            elementType: elementType
        }));
    }
    
    e.dataTransfer.effectAllowed = 'copy';
}

function handleDragEnd(e) {
    isDragging = false;
    draggedElement = null;
    e.target.classList.remove('dragging');
}

function handleDragOver(e) {
    e.preventDefault();
    e.dataTransfer.dropEffect = 'copy';
}

function handleDragEnter(e) {
    e.preventDefault();
    e.target.classList.add('drag-over');
}

function handleDragLeave(e) {
    if (!e.target.contains(e.relatedTarget)) {
        e.target.classList.remove('drag-over');
    }
}

function handleDrop(e) {
    e.preventDefault();
    e.target.classList.remove('drag-over');
    
    try {
        const jsonData = e.dataTransfer.getData('application/json');
        if (jsonData) {
            const data = JSON.parse(jsonData);
            
            if (data.type === 'variable') {
                insertVariable(data.variable, data.name, e);
            } else if (data.type === 'element') {
                insertElement(data.elementType, e);
            }
        }
    } catch (error) {
        console.error('Error handling drop:', error);
        showMessage('حدث خطأ أثناء إدراج العنصر', 'error');
    }
}

function insertVariable(variable, name, event) {
    const selection = window.getSelection();
    const range = document.createRange();
    
    // Create variable placeholder
    const placeholder = document.createElement('span');
    placeholder.className = 'variable-placeholder';
    placeholder.dataset.variable = variable;
    placeholder.textContent = `[${name}]`;
    placeholder.title = variable;
    
    // Insert at cursor position or at the end
    if (selection.rangeCount > 0 && elements.contractContent.contains(selection.anchorNode)) {
        const selectedRange = selection.getRangeAt(0);
        selectedRange.deleteContents();
        selectedRange.insertNode(placeholder);
        selectedRange.setStartAfter(placeholder);
        selectedRange.collapse(true);
        selection.removeAllRanges();
        selection.addRange(selectedRange);
    } else {
        elements.contractContent.appendChild(placeholder);
    }
    
    saveToHistory();
    showMessage(`تم إدراج المتغير: ${name}`, 'success');
}

function insertElement(elementType, event) {
    let elementHTML = '';
    
    switch (elementType) {
        case 'heading':
            elementHTML = '<h2 contenteditable="true">عنوان جديد</h2>';
            break;
        case 'paragraph':
            elementHTML = '<p contenteditable="true">فقرة جديدة. اكتب النص هنا...</p>';
            break;
        case 'list':
            elementHTML = `
                <ul contenteditable="true">
                    <li>العنصر الأول</li>
                    <li>العنصر الثاني</li>
                    <li>العنصر الثالث</li>
                </ul>
            `;
            break;
        case 'table':
            elementHTML = `
                <table border="1" style="width: 100%; border-collapse: collapse; margin: 1rem 0;">
                    <thead>
                        <tr>
                            <th style="padding: 0.5rem; background-color: #f8f9fa;">العمود الأول</th>
                            <th style="padding: 0.5rem; background-color: #f8f9fa;">العمود الثاني</th>
                            <th style="padding: 0.5rem; background-color: #f8f9fa;">العمود الثالث</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td style="padding: 0.5rem;" contenteditable="true">البيانات 1</td>
                            <td style="padding: 0.5rem;" contenteditable="true">البيانات 2</td>
                            <td style="padding: 0.5rem;" contenteditable="true">البيانات 3</td>
                        </tr>
                        <tr>
                            <td style="padding: 0.5rem;" contenteditable="true">البيانات 4</td>
                            <td style="padding: 0.5rem;" contenteditable="true">البيانات 5</td>
                            <td style="padding: 0.5rem;" contenteditable="true">البيانات 6</td>
                        </tr>
                    </tbody>
                </table>
            `;
            break;
        case 'signature':
            elementHTML = `
                <div style="margin: 2rem 0; display: flex; justify-content: space-between;">
                    <div style="text-align: center; width: 200px;">
                        <div style="border-top: 1px solid #000; margin-bottom: 10px;"></div>
                        <p contenteditable="true">توقيع الطرف الأول</p>
                        <p contenteditable="true">التاريخ: ___________</p>
                    </div>
                    <div style="text-align: center; width: 200px;">
                        <div style="border-top: 1px solid #000; margin-bottom: 10px;"></div>
                        <p contenteditable="true">توقيع الطرف الثاني</p>
                        <p contenteditable="true">التاريخ: ___________</p>
                    </div>
                </div>
            `;
            break;
        default:
            elementHTML = '<p contenteditable="true">عنصر جديد</p>';
    }
    
    // Insert element
    const selection = window.getSelection();
    if (selection.rangeCount > 0 && elements.contractContent.contains(selection.anchorNode)) {
        const range = selection.getRangeAt(0);
        const div = document.createElement('div');
        div.innerHTML = elementHTML;
        range.deleteContents();
        range.insertNode(div.firstChild);
    } else {
        elements.contractContent.insertAdjacentHTML('beforeend', elementHTML);
    }
    
    saveToHistory();
    showMessage(`تم إدراج عنصر: ${getElementTypeName(elementType)}`, 'success');
}

function getElementTypeName(type) {
    const names = {
        'heading': 'عنوان',
        'paragraph': 'فقرة',
        'list': 'قائمة',
        'table': 'جدول',
        'signature': 'منطقة توقيع'
    };
    return names[type] || type;
}

// Tab Functionality
function switchTab(tabName) {
    // Update tab buttons
    elements.tabBtns.forEach(btn => {
        btn.classList.toggle('active', btn.dataset.tab === tabName);
    });
    
    // Update tab content
    elements.tabContents.forEach(content => {
        content.classList.toggle('active', content.id === `${tabName}-tab`);
    });
}

// Search Functionality
function searchVariables() {
    const searchTerm = elements.variableSearch.value.toLowerCase();
    const variableItems = document.querySelectorAll('.variable-item');
    
    variableItems.forEach(item => {
        const variableCode = item.querySelector('.variable-code').textContent.toLowerCase();
        const variableName = item.querySelector('.variable-name').textContent.toLowerCase();
        const matches = variableCode.includes(searchTerm) || variableName.includes(searchTerm);
        
        item.style.display = matches ? 'flex' : 'none';
    });
    
    // Hide empty categories
    const categories = document.querySelectorAll('.variable-category');
    categories.forEach(category => {
        const visibleItems = category.querySelectorAll('.variable-item[style*="flex"], .variable-item:not([style*="none"])');
        category.style.display = visibleItems.length > 0 ? 'block' : 'none';
    });
}

// Content Management
function handleContentChange() {
    contractData.content = elements.contractContent.innerHTML;
    saveToHistory();
    updateToolbarState();
}

function handlePaste(e) {
    e.preventDefault();
    const text = e.clipboardData.getData('text/plain');
    document.execCommand('insertText', false, text);
}

// History Management
function saveToHistory() {
    const currentState = {
        content: elements.contractContent.innerHTML,
        timestamp: Date.now()
    };
    
    // Remove future history if we're not at the end
    if (contractData.historyIndex < contractData.history.length - 1) {
        contractData.history = contractData.history.slice(0, contractData.historyIndex + 1);
    }
    
    contractData.history.push(currentState);
    contractData.historyIndex = contractData.history.length - 1;
    
    // Limit history size
    if (contractData.history.length > 50) {
        contractData.history.shift();
        contractData.historyIndex--;
    }
    
    updateToolbarState();
}

function undo() {
    if (contractData.historyIndex > 0) {
        contractData.historyIndex--;
        const state = contractData.history[contractData.historyIndex];
        elements.contractContent.innerHTML = state.content;
        updateToolbarState();
        showMessage('تم التراجع', 'info');
    }
}

function redo() {
    if (contractData.historyIndex < contractData.history.length - 1) {
        contractData.historyIndex++;
        const state = contractData.history[contractData.historyIndex];
        elements.contractContent.innerHTML = state.content;
        updateToolbarState();
        showMessage('تم الإعادة', 'info');
    }
}

function updateToolbarState() {
    if (elements.undoBtn) {
        elements.undoBtn.disabled = contractData.historyIndex <= 0;
        elements.undoBtn.style.opacity = contractData.historyIndex <= 0 ? '0.5' : '1';
    }
    
    if (elements.redoBtn) {
        elements.redoBtn.disabled = contractData.historyIndex >= contractData.history.length - 1;
        elements.redoBtn.style.opacity = contractData.historyIndex >= contractData.history.length - 1 ? '0.5' : '1';
    }
}

// Contract Operations
function newContract() {
    if (confirm('هل أنت متأكد من إنشاء عقد جديد؟ سيتم فقدان التغييرات غير المحفوظة.')) {
        elements.contractContent.innerHTML = `
            <h1 style="text-align: center; margin-bottom: 30px;">عقد خدمات النقل</h1>
            
            <p>هذا عقد خدمات نقل مبرم بين:</p>
            
            <div style="margin: 20px 0;">
                <strong>الطرف الأول (الشركة):</strong><br>
                اسم الشركة: <span class="variable-placeholder" data-variable="$company_name">[اسم الشركة]</span><br>
                العنوان: <span class="variable-placeholder" data-variable="$company_address">[عنوان الشركة]</span><br>
                الهاتف: <span class="variable-placeholder" data-variable="$company_phone">[هاتف الشركة]</span>
            </div>
            
            <div style="margin: 20px 0;">
                <strong>الطرف الثاني (العميل):</strong><br>
                الاسم: <span class="variable-placeholder" data-variable="$client_name">[اسم العميل]</span><br>
                العنوان: <span class="variable-placeholder" data-variable="$client_address">[عنوان العميل]</span><br>
                الهاتف: <span class="variable-placeholder" data-variable="$client_phone">[هاتف العميل]</span>
            </div>
            
            <h2>تفاصيل العقد:</h2>
            <p>رقم العقد: <span class="variable-placeholder" data-variable="$contract_number">[رقم العقد]</span></p>
            <p>تاريخ العقد: <span class="variable-placeholder" data-variable="$contract_date">[تاريخ العقد]</span></p>
            <p>المبلغ الإجمالي: <span class="variable-placeholder" data-variable="$total_amount">[المبلغ الإجمالي]</span> <span class="variable-placeholder" data-variable="$currency">[العملة]</span></p>
            
            <div style="margin-top: 50px;">
                <div style="display: flex; justify-content: space-between;">
                    <div style="text-align: center;">
                        <div style="border-top: 1px solid #000; width: 200px; margin-bottom: 10px;"></div>
                        <p>توقيع العميل</p>
                        <p><span class="variable-placeholder" data-variable="$client_name">[اسم العميل]</span></p>
                    </div>
                    <div style="text-align: center;">
                        <div style="border-top: 1px solid #000; width: 200px; margin-bottom: 10px;"></div>
                        <p>توقيع الشركة</p>
                        <p><span class="variable-placeholder" data-variable="$company_name">[اسم الشركة]</span></p>
                    </div>
                </div>
            </div>
        `;
        
        contractData.history = [];
        contractData.historyIndex = -1;
        saveToHistory();
        showMessage('تم إنشاء عقد جديد', 'success');
    }
}

function openContract() {
    const input = document.createElement('input');
    input.type = 'file';
    input.accept = '.json,.html';
    input.onchange = function(e) {
        const file = e.target.files[0];
        if (file) {
            const reader = new FileReader();
            reader.onload = function(e) {
                try {
                    if (file.name.endsWith('.json')) {
                        const data = JSON.parse(e.target.result);
                        elements.contractContent.innerHTML = data.content || '';
                        contractData.variables = data.variables || {};
                    } else {
                        elements.contractContent.innerHTML = e.target.result;
                    }
                    saveToHistory();
                    showMessage('تم فتح العقد بنجاح', 'success');
                } catch (error) {
                    showMessage('خطأ في فتح الملف', 'error');
                }
            };
            reader.readAsText(file);
        }
    };
    input.click();
}

function saveContract() {
    const contractData = {
        content: elements.contractContent.innerHTML,
        variables: extractVariables(),
        timestamp: new Date().toISOString(),
        version: '1.0'
    };
    
    const blob = new Blob([JSON.stringify(contractData, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `contract_${new Date().toISOString().split('T')[0]}.json`;
    a.click();
    URL.revokeObjectURL(url);
    
    showMessage('تم حفظ العقد بنجاح', 'success');
}

function extractVariables() {
    const variables = {};
    const placeholders = elements.contractContent.querySelectorAll('.variable-placeholder');
    placeholders.forEach(placeholder => {
        const variable = placeholder.dataset.variable;
        if (variable) {
            variables[variable] = placeholder.textContent;
        }
    });
    return variables;
}

// Modal Functions
function showExportModal() {
    elements.exportModal.classList.add('active');
}

function hideExportModal() {
    elements.exportModal.classList.remove('active');
}

function showPreview() {
    const content = elements.contractContent.innerHTML;
    elements.previewContent.innerHTML = content;
    elements.previewModal.classList.add('active');
}

function hidePreviewModal() {
    elements.previewModal.classList.remove('active');
}

// Export Functions
function exportContract(format) {
    const content = elements.contractContent.innerHTML;
    const variables = extractVariables();
    
    switch (format) {
        case 'php':
            exportToPHP(content, variables);
            break;
        case 'html':
            exportToHTML(content);
            break;
        case 'pdf':
            exportToPDF(content);
            break;
        case 'json':
            exportToJSON(content, variables);
            break;
        default:
            showMessage('تنسيق التصدير غير مدعوم', 'error');
    }
    
    hideExportModal();
}

function exportToPHP(content, variables) {
    const phpCode = `<?php
/**
 * Contract Template - Generated by Contract Editor
 * Date: ${new Date().toISOString()}
 */

// Contract Variables
${Object.entries(variables).map(([key, value]) => `${key} = "${value}";`).join('\n')}

// Contract Content
function generateContract($variables = []) {
    // Extract variables
    ${Object.keys(variables).map(key => `$${key.substring(1)} = $variables['${key}'] ?? '${variables[key]}';`).join('\n    ')}
    
    $html = '${content.replace(/'/g, "\\'")}';
    
    // Replace variable placeholders
    ${Object.keys(variables).map(key => `$html = str_replace('[${variables[key]}]', $${key.substring(1)}, $html);`).join('\n    ')}
    
    return $html;
}

// Usage Example
$contractVariables = [
    ${Object.entries(variables).map(([key, value]) => `'${key}' => '${value}'`).join(',\n    ')}
];

$contractHTML = generateContract($contractVariables);

// Save to database (example)
function saveContractToDatabase($businessId, $locationId, $contractHTML, $variables) {
    // Database connection
    $pdo = new PDO('mysql:host=localhost;dbname=your_database', $username, $password);
    
    $sql = "INSERT INTO contracts (business_id, location_id, contract_html, contract_data, created_at) 
            VALUES (:business_id, :location_id, :contract_html, :contract_data, NOW())";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        ':business_id' => $businessId,
        ':location_id' => $locationId,
        ':contract_html' => $contractHTML,
        ':contract_data' => json_encode($variables)
    ]);
    
    return $pdo->lastInsertId();
}

// Display contract
echo $contractHTML;
?>`;
    
    downloadFile(phpCode, 'contract_template.php', 'text/plain');
    showMessage('تم تصدير ملف PHP بنجاح', 'success');
}

function exportToHTML(content) {
    const htmlContent = `<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>عقد خدمات النقل</title>
    <style>
        body {
            font-family: 'Cairo', Arial, sans-serif;
            direction: rtl;
            max-width: 800px;
            margin: 0 auto;
            padding: 2rem;
            line-height: 1.8;
            color: #333;
        }
        .variable-placeholder {
            background-color: rgba(37, 99, 235, 0.1);
            color: #2563eb;
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
            font-weight: 600;
            border: 1px dashed #2563eb;
        }
        h1, h2 {
            color: #1e293b;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 1rem 0;
        }
        th, td {
            padding: 0.5rem;
            border: 1px solid #e2e8f0;
            text-align: right;
        }
        th {
            background-color: #f8f9fa;
            font-weight: 600;
        }
        @media print {
            body {
                padding: 0;
            }
        }
    </style>
</head>
<body>
    ${content}
</body>
</html>`;
    
    downloadFile(htmlContent, 'contract.html', 'text/html');
    showMessage('تم تصدير ملف HTML بنجاح', 'success');
}

function exportToPDF(content) {
    // Create a new window for printing
    const printWindow = window.open('', '_blank');
    printWindow.document.write(`
        <!DOCTYPE html>
        <html lang="ar" dir="rtl">
        <head>
            <meta charset="UTF-8">
            <title>عقد خدمات النقل</title>
            <style>
                body {
                    font-family: 'Cairo', Arial, sans-serif;
                    direction: rtl;
                    max-width: 800px;
                    margin: 0 auto;
                    padding: 2rem;
                    line-height: 1.8;
                    color: #333;
                }
                .variable-placeholder {
                    background-color: rgba(37, 99, 235, 0.1);
                    color: #2563eb;
                    padding: 0.25rem 0.5rem;
                    border-radius: 4px;
                    font-weight: 600;
                    border: 1px dashed #2563eb;
                }
                h1, h2 {
                    color: #1e293b;
                }
                table {
                    width: 100%;
                    border-collapse: collapse;
                    margin: 1rem 0;
                }
                th, td {
                    padding: 0.5rem;
                    border: 1px solid #e2e8f0;
                    text-align: right;
                }
                th {
                    background-color: #f8f9fa;
                    font-weight: 600;
                }
                @media print {
                    body {
                        padding: 0;
                    }
                }
            </style>
        </head>
        <body>
            ${content}
        </body>
        </html>
    `);
    
    printWindow.document.close();
    printWindow.focus();
    
    setTimeout(() => {
        printWindow.print();
        printWindow.close();
    }, 500);
    
    showMessage('تم فتح نافذة الطباعة لحفظ PDF', 'info');
}

function exportToJSON(content, variables) {
    const jsonData = {
        content: content,
        variables: variables,
        metadata: {
            exportDate: new Date().toISOString(),
            version: '1.0',
            editor: 'Contract Editor Pro'
        }
    };
    
    downloadFile(JSON.stringify(jsonData, null, 2), 'contract_data.json', 'application/json');
    showMessage('تم تصدير ملف JSON بنجاح', 'success');
}

function downloadFile(content, filename, mimeType) {
    const blob = new Blob([content], { type: mimeType });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    a.click();
    URL.revokeObjectURL(url);
}

// Keyboard Shortcuts
function handleKeyboardShortcuts(e) {
    if (e.ctrlKey || e.metaKey) {
        switch (e.key) {
            case 'n':
                e.preventDefault();
                newContract();
                break;
            case 'o':
                e.preventDefault();
                openContract();
                break;
            case 's':
                e.preventDefault();
                saveContract();
                break;
            case 'z':
                e.preventDefault();
                if (e.shiftKey) {
                    redo();
                } else {
                    undo();
                }
                break;
            case 'y':
                e.preventDefault();
                redo();
                break;
            case 'p':
                e.preventDefault();
                showPreview();
                break;
            case 'e':
                e.preventDefault();
                showExportModal();
                break;
        }
    }
}

// Utility Functions
function showMessage(message, type = 'info') {
    // Remove existing messages
    const existingMessages = document.querySelectorAll('.message');
    existingMessages.forEach(msg => msg.remove());
    
    // Create new message
    const messageDiv = document.createElement('div');
    messageDiv.className = `message message-${type}`;
    messageDiv.innerHTML = `
        <i class="fas fa-${getMessageIcon(type)}"></i>
        <span>${message}</span>
    `;
    
    // Insert message
    const header = document.querySelector('.header');
    header.insertAdjacentElement('afterend', messageDiv);
    
    // Auto remove after 3 seconds
    setTimeout(() => {
        messageDiv.remove();
    }, 3000);
}

function getMessageIcon(type) {
    const icons = {
        'success': 'check-circle',
        'error': 'exclamation-circle',
        'info': 'info-circle',
        'warning': 'exclamation-triangle'
    };
    return icons[type] || 'info-circle';
}

// Auto-save functionality
setInterval(() => {
    if (elements.contractContent && elements.contractContent.innerHTML.trim()) {
        localStorage.setItem('contract_autosave', JSON.stringify({
            content: elements.contractContent.innerHTML,
            timestamp: Date.now()
        }));
    }
}, 30000); // Auto-save every 30 seconds

// Load auto-saved content on page load
window.addEventListener('load', () => {
    const autosaved = localStorage.getItem('contract_autosave');
    if (autosaved) {
        try {
            const data = JSON.parse(autosaved);
            const timeDiff = Date.now() - data.timestamp;
            
            // If auto-saved content is less than 1 hour old
            if (timeDiff < 3600000) {
                if (confirm('تم العثور على محتوى محفوظ تلقائياً. هل تريد استعادته؟')) {
                    elements.contractContent.innerHTML = data.content;
                    saveToHistory();
                    showMessage('تم استعادة المحتوى المحفوظ تلقائياً', 'info');
                }
            }
        } catch (error) {
            console.error('Error loading auto-saved content:', error);
        }
    }
});

// Export global functions for debugging
window.ContractEditor = {
    newContract,
    openContract,
    saveContract,
    exportContract,
    showPreview,
    undo,
    redo,
    switchTab,
    searchVariables
};

